/*
  short.c
  Copyright 2000, Seth Golub <seth@aigeek.com>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307,
  USA.
*/

#include <stdio.h>
#include <stdlib.h>
#include <sndfile.h>
#include <rfftw.h>
#include <math.h>
#include "short.h"
#include "array.h"
#include "error.h"

void init_s_workspace( s_workspace *sw, SF_INFO *sfinfo )
{
  int i;
  float log2 = log( 2 );

  sw->N = sfinfo->samplerate / SWPS;

  sw->signal_int    = new_int_array( sw->N * sfinfo->channels );
  sw->signal_real   = new_real_array( sw->N );
  sw->freq          = new_real_array( sw->N );
  sw->power         = new_real_array( (sw->N/2 + 1) );
  sw->logscale      = new_real_array( (sw->N/2 + 1) );

  sw->uniformity_scale = log( sw->power->size );
  sw->logscale_sum = 0.0;
  sw->inverse_logscale_sum = 0.0;
  for ( i=0; i < sw->logscale->size; i++ )
    {
      sw->logscale->data[i] = log(i+2) / log2;
      sw->logscale_sum += sw->logscale->data[i];
      sw->inverse_logscale_sum += 1.0 / sw->logscale->data[i];
    }
  
  sw->plan = rfftw_create_plan( sw->N, FFTW_REAL_TO_COMPLEX, FFTW_ESTIMATE );

  if ( !sw->plan )
    {
      fprintf( stderr, "Out of memory.\n" );
      exit( ERR_MEM );
    }
}

void free_s_workspace( s_workspace *sw )
{
  free_int_array( sw->signal_int );
  free_real_array( sw->signal_real );
  free_real_array( sw->freq );
  free_real_array( sw->power );
  free_real_array( sw->logscale );
  rfftw_destroy_plan( sw->plan );
}


/* 
 * Copies a 1-channel signal from an int array to a real array,
 * or mixes a 2-channel stereo signal from int to real.
 * signal_int contains <samples> * <channels> items
 * signal_real must be at least <samples> items long.
 *
 * Assumes channels is 1 or 2.  (This must be ensured elsewhere.)
 */
void mix_to_real( int_array signal_int, real_array signal_real, 
                  int channels )
{
  int i;
  if ( channels == 1 )
    for ( i=0; i < signal_int->size; i++ )
      signal_real->data[i] = signal_int->data[i];
  else
    for ( i=0; i < signal_real->size; i++ )
      signal_real->data[i] = (signal_int->data[i*2] 
                              + signal_int->data[i*2+1]) / 2;
}


/* power spectrum code snippet from FFTW docs */
void find_power_spectrum( real_array freq, real_array power )
{
  int i, N;
  N = freq->size;
  power->data[0] = freq->data[0]*freq->data[0];  /* DC component */
  for (i = 1; i < (N+1)/2; ++i)
    power->data[i] =(freq->data[i]*freq->data[i] + freq->data[N-i]*freq->data[N-i]);
  if (N % 2 == 0)
    power->data[N/2] = freq->data[N/2]*freq->data[N/2];
}

/*
  centroid:  power-weighted mean, expressed in the logscale * 1000.
  bandwidth: power-weighted std deviation, also on the logscale * 1000.
  uniformity:  negative entropy of frequency, not on the logscale.  range 0-1000.
*/
void find_freq_stats( s_workspace *sw, int *centroidp, int *bandwidthp, int *uniformityp )
{
  int i;
  double centroid;
  double sum = 0.0, weighted_sum = 0.0;
  double uniformity;
  for ( i=0; i < sw->power->size; i++ )
    {
      sum                     += sw->power->data[i];
      weighted_sum            += sw->power->data[i] * sw->logscale->data[i];
    }

  centroid = weighted_sum / sum;
  
  weighted_sum = 0.0;
  uniformity = 0.0;
  for ( i=0; i < sw->power->size; i++ )
    {
      if ( sw->power->data[i] > 0 )
        uniformity += (sw->power->data[i] / sum) * log(sw->power->data[i] / sum);
      weighted_sum += ( (centroid - sw->logscale->data[i])
                       * (centroid - sw->logscale->data[i])
                       * sw->power->data[i] );
    }
  *centroidp  = (int) (1000.0 * centroid);
  *bandwidthp = (int) (1000.0 * sqrt(weighted_sum / sum));
  *uniformityp  = (int) (-1000.0 * uniformity / sw->uniformity_scale);
}


/* Reads data from input file and calculates short window statistics.
 * Returns nonzero iff a full window could be read.  */
int find_short_stats( workspace *w, int *loudnessp, int *centroidp, 
                      int *bandwidthp, int *uniformityp )
{
  if ( sf_readf_int( w->inputFile, w->sw.signal_int->data, w->sw.N ) < w->sw.N )
    return 0;
  
  *loudnessp = mean_log2_abs( w->sw.signal_int );
  if ( *loudnessp == 0 )
    {
      *centroidp = 0; /* These won't matter, but we need to record something. */
      *bandwidthp = 0;
      *uniformityp = 0;
    }
  else
    {
      mix_to_real( w->sw.signal_int, w->sw.signal_real, w->sfinfo->channels );
      rfftw_one( w->sw.plan, w->sw.signal_real->data, w->sw.freq->data );
      find_power_spectrum( w->sw.freq, w->sw.power );
      find_freq_stats( &(w->sw), centroidp, bandwidthp, uniformityp );
    }
  return 1;
}
