/*
  medium.c
  Copyright 2000, Seth Golub <seth@aigeek.com>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307,
  USA.
*/

#include <math.h>
#include "short.h"
#include "medium.h"

/*
 * Reads data from input file and calculates medium window statistics.
 * Returns nonzero iff a full window could be read.
 */
int collect_short_stats( workspace *w )
{
  int i, centroid, bandwidth, loudness, uniformity;

  for ( i=0; (i < SWPMW) && find_short_stats( w, &loudness, &centroid, &bandwidth, &uniformity ); i++ )
    {
      w->mw.loudness->data[i] = loudness;
      w->mw.centroid->data[i] = centroid;
      w->mw.bandwidth->data[i] = bandwidth;
      w->mw.uniformity->data[i] = uniformity;
      if ( i != 0 )
        {
          w->mw.loudness_diff->data[i-1]    = loudness    - w->mw.loudness_diff->data[i-2];
          w->mw.centroid_diff->data[i-1]  = centroid  - w->mw.centroid_diff->data[i-2];
          w->mw.bandwidth_diff->data[i-1] = bandwidth - w->mw.bandwidth_diff->data[i-2];
          w->mw.uniformity_diff->data[i-1]  = uniformity  - w->mw.uniformity_diff->data[i-2];
        }
    }
  return (i == SWPMW);
}

int find_medium_stats( workspace *w, m_stats *mstat )
{
  long weightsum;
  if ( !collect_short_stats( w ) )
    return 0;

  mstat->loudness_mean         = mean( w->mw.loudness );
  mstat->loudness_std          = std( w->mw.loudness, mstat->loudness_mean );
  mstat->centroid_mean       = mean( w->mw.centroid );
  mstat->centroid_std        = std( w->mw.centroid, mstat->centroid_mean );
  mstat->bandwidth_mean      = mean( w->mw.bandwidth );
  mstat->bandwidth_std       = std( w->mw.bandwidth, mstat->bandwidth_mean );
  mstat->uniformity_mean      = mean( w->mw.uniformity );
  mstat->uniformity_std       = std( w->mw.uniformity, mstat->uniformity_mean );
  mstat->loudness_diff_mean    = mean( w->mw.loudness_diff );
  mstat->loudness_diff_std     = std( w->mw.loudness_diff, mstat->loudness_diff_mean );
  mstat->centroid_diff_mean  = mean( w->mw.centroid_diff );
  mstat->centroid_diff_std   = std( w->mw.centroid_diff, mstat->centroid_diff_mean );
  mstat->bandwidth_diff_mean = mean( w->mw.bandwidth_diff );
  mstat->bandwidth_diff_std  = std( w->mw.bandwidth_diff, mstat->bandwidth_diff_mean );
  mstat->uniformity_diff_mean = mean( w->mw.uniformity_diff );
  mstat->uniformity_diff_std  = std( w->mw.uniformity_diff, mstat->uniformity_diff_mean );
  
  weightsum = sum( w->mw.loudness );
  if ( weightsum == 0 )
    {
      mstat->centroid_wmean  = 0.0;
      mstat->centroid_wstd   = 0.0;
      mstat->bandwidth_wmean = 0.0;
      mstat->bandwidth_wstd  = 0.0;
      mstat->uniformity_wmean  = 0.0;
      mstat->uniformity_wstd   = 0.0;
    }
  else
    {
      mstat->centroid_wmean  = wmean( w->mw.centroid, w->mw.loudness, weightsum );
      mstat->centroid_wstd   = wstd( w->mw.centroid, mstat->centroid_wmean, w->mw.loudness, weightsum );
      mstat->bandwidth_wmean = wmean( w->mw.bandwidth, w->mw.loudness, weightsum );
      mstat->bandwidth_wstd  = wstd( w->mw.bandwidth, mstat->bandwidth_wmean, w->mw.loudness, weightsum );
      mstat->uniformity_wmean  = wmean( w->mw.uniformity, w->mw.loudness, weightsum );
      mstat->uniformity_wstd   = wstd( w->mw.uniformity, mstat->uniformity_wmean, w->mw.loudness, weightsum );
    }
  return 1;
}


/*
 * The m_workspace is to hold arrays for the medium window analysis.
 */
void init_m_workspace( m_workspace *mw )
{
  mw->loudness         = new_int_array( SWPMW );
  mw->centroid       = new_int_array( SWPMW );
  mw->bandwidth      = new_int_array( SWPMW );
  mw->uniformity       = new_int_array( SWPMW );
  mw->loudness_diff    = new_int_array( SWPMW - 1 );
  mw->centroid_diff  = new_int_array( SWPMW - 1 );
  mw->bandwidth_diff = new_int_array( SWPMW - 1 );
  mw->uniformity_diff  = new_int_array( SWPMW - 1 );
}

void free_m_workspace( m_workspace *mw )
{
  free_int_array( mw->loudness );
  free_int_array( mw->centroid );
  free_int_array( mw->bandwidth );
  free_int_array( mw->uniformity );
  free_int_array( mw->loudness_diff );
  free_int_array( mw->centroid_diff );
  free_int_array( mw->bandwidth_diff );
  free_int_array( mw->uniformity_diff );
}

