/*
  long.c
  Copyright 2000, Seth Golub <seth@aigeek.com>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307,
  USA.
*/

#include <math.h>
#include "short.h"
#include "medium.h"
#include "long.h"

void collect_medium_stats( workspace *w )
{
  int i;
  m_stats mstats;

  for ( i=0; find_medium_stats( w, &mstats ); i++ )
    {
      w->lw.loudness_mean->data[i] = mstats.loudness_mean;
      w->lw.loudness_std->data[i] = mstats.loudness_std;

      w->lw.centroid_mean->data[i] = mstats.centroid_mean;
      w->lw.centroid_std->data[i] = mstats.centroid_std;
      w->lw.bandwidth_mean->data[i] = mstats.bandwidth_mean;
      w->lw.bandwidth_std->data[i] = mstats.bandwidth_std;
      w->lw.uniformity_mean->data[i] = mstats.uniformity_mean;
      w->lw.uniformity_std->data[i] = mstats.uniformity_std;

      w->lw.centroid_wmean->data[i] = mstats.centroid_wmean;
      w->lw.centroid_wstd->data[i] = mstats.centroid_wstd;
      w->lw.bandwidth_wmean->data[i] = mstats.bandwidth_wmean;
      w->lw.bandwidth_wstd->data[i] = mstats.bandwidth_wstd;
      w->lw.uniformity_wmean->data[i] = mstats.uniformity_wmean;
      w->lw.uniformity_wstd->data[i] = mstats.uniformity_wstd;

      w->lw.loudness_diff_mean->data[i] = mstats.loudness_diff_mean;
      w->lw.loudness_diff_std->data[i] = mstats.loudness_diff_std;
      w->lw.centroid_diff_mean->data[i] = mstats.centroid_diff_mean;
      w->lw.centroid_diff_std->data[i] = mstats.centroid_diff_std;
      w->lw.bandwidth_diff_mean->data[i] = mstats.bandwidth_diff_mean;
      w->lw.bandwidth_diff_std->data[i] = mstats.bandwidth_diff_std;
      w->lw.uniformity_diff_mean->data[i] = mstats.uniformity_diff_mean;
      w->lw.uniformity_diff_std->data[i] = mstats.uniformity_diff_std;
    }
}

float loudness_scale_factor( workspace *w )
{
  float max=0.0;
  int i;
  float frame_max;
  for ( i=0; i < w->lw.loudness_mean->size; i++ )
    {
      frame_max = w->lw.loudness_mean->data[i] + w->lw.loudness_std->data[i];
      if ( frame_max > max )
        max = frame_max;
    }
  return max / 1000.0;
}


void find_long_stats( workspace *w, l_stats *lstat )
{
  collect_medium_stats( w );

  lstat->length = (w->sfinfo->samples / w->sfinfo->samplerate);
  lstat->loudness_scale_factor = loudness_scale_factor( w );

  if ( lstat->loudness_scale_factor < 0.001 ) /* Silent sample? */
    {
      /* This introduces a discontinuity in the scaled features, but
         they're all garbage at this loudness level anyway. */
      lstat->loudness_scale_factor = 1.0;  /* We don't want errors, nor do we
                                            want to scale up tiny noise. */
    }

  lstat->loudness_mean_mean = meanf( w->lw.loudness_mean ) / lstat->loudness_scale_factor;
  lstat->loudness_mean_std = stdf( w->lw.loudness_mean, lstat->loudness_mean_mean ) / lstat->loudness_scale_factor;
  lstat->loudness_std_mean = meanf( w->lw.loudness_std ) / lstat->loudness_scale_factor;
  lstat->loudness_std_std = stdf( w->lw.loudness_std, lstat->loudness_std_mean ) / lstat->loudness_scale_factor;

  lstat->centroid_mean_mean = meanf( w->lw.centroid_mean );
  lstat->centroid_mean_std = stdf( w->lw.centroid_mean, lstat->centroid_mean_mean );
  lstat->centroid_std_mean = meanf( w->lw.centroid_std );
  lstat->centroid_std_std = stdf( w->lw.centroid_std, lstat->centroid_std_mean );

  lstat->bandwidth_mean_mean = meanf( w->lw.bandwidth_mean );
  lstat->bandwidth_mean_std = stdf( w->lw.bandwidth_mean, lstat->bandwidth_mean_mean );
  lstat->bandwidth_std_mean = meanf( w->lw.bandwidth_std );
  lstat->bandwidth_std_std = stdf( w->lw.bandwidth_std, lstat->bandwidth_std_mean );

  lstat->uniformity_mean_mean = meanf( w->lw.uniformity_mean );
  lstat->uniformity_mean_std = stdf( w->lw.uniformity_mean, lstat->uniformity_mean_mean );
  lstat->uniformity_std_mean = meanf( w->lw.uniformity_std );
  lstat->uniformity_std_std = stdf( w->lw.uniformity_std, lstat->uniformity_std_mean );

  lstat->centroid_wmean_mean = meanf( w->lw.centroid_wmean );
  lstat->centroid_wmean_std = stdf( w->lw.centroid_wmean, lstat->centroid_wmean_mean );
  lstat->centroid_wstd_mean = meanf( w->lw.centroid_wstd );
  lstat->centroid_wstd_std = stdf( w->lw.centroid_wstd, lstat->centroid_wstd_mean );

  lstat->bandwidth_wmean_mean = meanf( w->lw.bandwidth_wmean );
  lstat->bandwidth_wmean_std = stdf( w->lw.bandwidth_wmean, lstat->bandwidth_wmean_mean );
  lstat->bandwidth_wstd_mean = meanf( w->lw.bandwidth_wstd );
  lstat->bandwidth_wstd_std = stdf( w->lw.bandwidth_wstd, lstat->bandwidth_wstd_mean );

  lstat->uniformity_wmean_mean = meanf( w->lw.uniformity_wmean );
  lstat->uniformity_wmean_std = stdf( w->lw.uniformity_wmean, lstat->uniformity_wmean_mean );
  lstat->uniformity_wstd_mean = meanf( w->lw.uniformity_wstd );
  lstat->uniformity_wstd_std = stdf( w->lw.uniformity_wstd, lstat->uniformity_wstd_mean );

  lstat->loudness_diff_mean_mean = meanf( w->lw.loudness_diff_mean ) 
                                     / lstat->loudness_scale_factor;
  lstat->loudness_diff_mean_std = stdf( w->lw.loudness_diff_mean, lstat->loudness_diff_mean_mean ) 
                                     / lstat->loudness_scale_factor;
  lstat->loudness_diff_std_mean = meanf( w->lw.loudness_diff_std ) / lstat->loudness_scale_factor;
  lstat->loudness_diff_std_std = stdf( w->lw.loudness_diff_std, lstat->loudness_diff_std_mean ) 
                                     / lstat->loudness_scale_factor;

  lstat->centroid_diff_mean_mean = meanf( w->lw.centroid_diff_mean );
  lstat->centroid_diff_mean_std = stdf( w->lw.centroid_diff_mean, lstat->centroid_diff_mean_mean );
  lstat->centroid_diff_std_mean = meanf( w->lw.centroid_diff_std );
  lstat->centroid_diff_std_std = stdf( w->lw.centroid_diff_std, lstat->centroid_diff_std_mean );

  lstat->bandwidth_diff_mean_mean = meanf( w->lw.bandwidth_diff_mean );
  lstat->bandwidth_diff_mean_std = stdf( w->lw.bandwidth_diff_mean, lstat->bandwidth_diff_mean_mean );
  lstat->bandwidth_diff_std_mean = meanf( w->lw.bandwidth_diff_std );
  lstat->bandwidth_diff_std_std = stdf( w->lw.bandwidth_diff_std, lstat->bandwidth_diff_std_mean );

  lstat->uniformity_diff_mean_mean = meanf( w->lw.uniformity_diff_mean );
  lstat->uniformity_diff_mean_std = stdf( w->lw.uniformity_diff_mean, lstat->uniformity_diff_mean_mean );
  lstat->uniformity_diff_std_mean = meanf( w->lw.uniformity_diff_std );
  lstat->uniformity_diff_std_std = stdf( w->lw.uniformity_diff_std, lstat->uniformity_diff_std_mean );
}


/*
 * The m_workspace is to hold arrays for the medium window analysis.
 */
void init_l_workspace( l_workspace *lw, SF_INFO *sfinfo )
{
  int N = sfinfo->samples / (sfinfo->samplerate / SWPS) / SWPMW;
  lw->loudness_mean         = new_float_array( N );
  lw->loudness_std          = new_float_array( N );

  lw->centroid_mean       = new_float_array( N );
  lw->centroid_std        = new_float_array( N );
  lw->bandwidth_mean      = new_float_array( N );
  lw->bandwidth_std       = new_float_array( N );
  lw->uniformity_mean       = new_float_array( N );
  lw->uniformity_std        = new_float_array( N );

  lw->centroid_wmean       = new_float_array( N );
  lw->centroid_wstd        = new_float_array( N );
  lw->bandwidth_wmean      = new_float_array( N );
  lw->bandwidth_wstd       = new_float_array( N );
  lw->uniformity_wmean       = new_float_array( N );
  lw->uniformity_wstd        = new_float_array( N );

  lw->loudness_diff_mean    = new_float_array( N );
  lw->loudness_diff_std     = new_float_array( N );
  lw->centroid_diff_mean  = new_float_array( N );
  lw->centroid_diff_std   = new_float_array( N );
  lw->bandwidth_diff_mean = new_float_array( N );
  lw->bandwidth_diff_std  = new_float_array( N );
  lw->uniformity_diff_mean  = new_float_array( N );
  lw->uniformity_diff_std   = new_float_array( N );
}

void free_l_workspace( l_workspace *lw )
{
  free_float_array( lw->loudness_mean );
  free_float_array( lw->loudness_std );

  free_float_array( lw->centroid_mean );
  free_float_array( lw->centroid_std );
  free_float_array( lw->bandwidth_mean );
  free_float_array( lw->bandwidth_std );
  free_float_array( lw->uniformity_mean );
  free_float_array( lw->uniformity_std );

  free_float_array( lw->centroid_wmean );
  free_float_array( lw->centroid_wstd );
  free_float_array( lw->bandwidth_wmean );
  free_float_array( lw->bandwidth_wstd );
  free_float_array( lw->uniformity_wmean );
  free_float_array( lw->uniformity_wstd );

  free_float_array( lw->loudness_diff_mean );
  free_float_array( lw->loudness_diff_std );
  free_float_array( lw->centroid_diff_mean );
  free_float_array( lw->centroid_diff_std );
  free_float_array( lw->bandwidth_diff_mean );
  free_float_array( lw->bandwidth_diff_std );
  free_float_array( lw->uniformity_diff_mean );
  free_float_array( lw->uniformity_diff_std );
}

/* Need to make shorter lines to print dissertation */
#define CONCAT(a,b) a##b

void print_long_stats( l_stats *lstat )
{
  printf( CONCAT("%d %f %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d",
                 " %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d %d\n"),
          lstat->length, lstat->loudness_scale_factor,

          lstat->loudness_mean_mean, lstat->loudness_mean_std,
          lstat->loudness_std_mean, lstat->loudness_std_std,

          lstat->centroid_mean_mean, lstat->centroid_mean_std,
          lstat->centroid_std_mean, lstat->centroid_std_std,
          lstat->bandwidth_mean_mean, lstat->bandwidth_mean_std,
          lstat->bandwidth_std_mean, lstat->bandwidth_std_std,
          lstat->uniformity_mean_mean, lstat->uniformity_mean_std,
          lstat->uniformity_std_mean, lstat->uniformity_std_std,

          lstat->centroid_wmean_mean, lstat->centroid_wmean_std,
          lstat->centroid_wstd_mean, lstat->centroid_wstd_std,
          lstat->bandwidth_wmean_mean, lstat->bandwidth_wmean_std,
          lstat->bandwidth_wstd_mean, lstat->bandwidth_wstd_std,
          lstat->uniformity_wmean_mean, lstat->uniformity_wmean_std,
          lstat->uniformity_wstd_mean, lstat->uniformity_wstd_std,

          lstat->loudness_diff_mean_mean, lstat->loudness_diff_mean_std,
          lstat->loudness_diff_std_mean, lstat->loudness_diff_std_std,
          lstat->centroid_diff_mean_mean, lstat->centroid_diff_mean_std,
          lstat->centroid_diff_std_mean, lstat->centroid_diff_std_std,
          lstat->bandwidth_diff_mean_mean, lstat->bandwidth_diff_mean_std,
          lstat->bandwidth_diff_std_mean, lstat->bandwidth_diff_std_std,
          lstat->uniformity_diff_mean_mean, lstat->uniformity_diff_mean_std,
          lstat->uniformity_diff_std_mean, lstat->uniformity_diff_std_std );
}

void print_long_stats_verbose( l_stats *lstat )
{
  printf( "    length: %d
    loudness_scale_factor: %f
    loudness_mean_mean: %d
    loudness_mean_std: %d
    loudness_std_mean: %d
    loudness_std_std: %d
    centroid_mean_mean: %d
    centroid_mean_std: %d
    centroid_std_mean: %d
    centroid_std_std: %d
    bandwidth_mean_mean: %d
    bandwidth_mean_std: %d
    bandwidth_std_mean: %d
    bandwidth_std_std: %d
    uniformity_mean_mean: %d
    uniformity_mean_std: %d
    uniformity_std_mean: %d
    uniformity_std_std: %d
    centroid_wmean_mean: %d
    centroid_wmean_std: %d
    centroid_wstd_mean: %d
    centroid_wstd_std: %d
    bandwidth_wmean_mean: %d
    bandwidth_wmean_std: %d
    bandwidth_wstd_mean: %d
    bandwidth_wstd_std: %d
    uniformity_wmean_mean: %d
    uniformity_wmean_std: %d
    uniformity_wstd_mean: %d
    uniformity_wstd_std: %d
    loudness_diff_mean_mean: %d
    loudness_diff_mean_std: %d
    loudness_diff_std_mean: %d
    loudness_diff_std_std: %d
    centroid_diff_mean_mean: %d
    centroid_diff_mean_std: %d
    centroid_diff_std_mean: %d
    centroid_diff_std_std: %d
    bandwidth_diff_mean_mean: %d
    bandwidth_diff_mean_std: %d
    bandwidth_diff_std_mean: %d
    bandwidth_diff_std_std: %d
    uniformity_diff_mean_mean: %d
    uniformity_diff_mean_std: %d
    uniformity_diff_std_mean: %d
    uniformity_diff_std_std: %d\n",
          lstat->length, lstat->loudness_scale_factor,

          lstat->loudness_mean_mean, lstat->loudness_mean_std,
          lstat->loudness_std_mean, lstat->loudness_std_std,

          lstat->centroid_mean_mean, lstat->centroid_mean_std,
          lstat->centroid_std_mean, lstat->centroid_std_std,
          lstat->bandwidth_mean_mean, lstat->bandwidth_mean_std,
          lstat->bandwidth_std_mean, lstat->bandwidth_std_std,
          lstat->uniformity_mean_mean, lstat->uniformity_mean_std,
          lstat->uniformity_std_mean, lstat->uniformity_std_std,

          lstat->centroid_wmean_mean, lstat->centroid_wmean_std,
          lstat->centroid_wstd_mean, lstat->centroid_wstd_std,
          lstat->bandwidth_wmean_mean, lstat->bandwidth_wmean_std,
          lstat->bandwidth_wstd_mean, lstat->bandwidth_wstd_std,
          lstat->uniformity_wmean_mean, lstat->uniformity_wmean_std,
          lstat->uniformity_wstd_mean, lstat->uniformity_wstd_std,

          lstat->loudness_diff_mean_mean, lstat->loudness_diff_mean_std,
          lstat->loudness_diff_std_mean, lstat->loudness_diff_std_std,
          lstat->centroid_diff_mean_mean, lstat->centroid_diff_mean_std,
          lstat->centroid_diff_std_mean, lstat->centroid_diff_std_std,
          lstat->bandwidth_diff_mean_mean, lstat->bandwidth_diff_mean_std,
          lstat->bandwidth_diff_std_mean, lstat->bandwidth_diff_std_std,
          lstat->uniformity_diff_mean_mean, lstat->uniformity_diff_mean_std,
          lstat->uniformity_diff_std_mean, lstat->uniformity_diff_std_std );
}
