/*
  array.c
  Copyright 2000, Seth Golub <seth@aigeek.com>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
  General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307,
  USA.
*/

#include <math.h>
#include "array.h"
#include "error.h"

void massert( void *x )
{
  if ( x == NULL )
    {
      fprintf( stderr, "Out of memory.\n" );
      exit( ERR_MEM );
    }
}

int_array new_int_array( int size )
{
  int_array arr = (int_array) malloc( sizeof(struct int_array_struct) );
  massert( arr );
  arr->size = size;
  arr->data = (int *) malloc( sizeof(int) * size );
  massert( arr->data );
  return arr;
}

real_array new_real_array( int size )
{
  real_array arr = (real_array) malloc( sizeof(struct real_array_struct) );
  massert( arr );
  arr->size = size;
  arr->data = (fftw_real *) malloc( sizeof(fftw_real) * size );
  massert( arr->data );
  return arr;
}

float_array new_float_array( int size )
{
  float_array arr = (float_array) malloc( sizeof(struct float_array_struct) );
  massert( arr );
  arr->size = size;
  arr->data = (float *) malloc( sizeof(float) * size );
  massert( arr->data );
  return arr;
}


void free_int_array( int_array arr )
{
  free( arr->data );
  free( arr );
}

void free_real_array( real_array arr )
{
  free( arr->data );
  free( arr );
}

void free_float_array( float_array arr )
{
  free( arr->data );
  free( arr );
}


/* Finds 100 * mean of log2 of absolute values of array elements
 * Useful for finding the mean loudness of the (rectified) signal
 */
int mean_log2_abs( int_array array )
{
  static double log2 = 0;
  int i, sum = 0;
  if ( log2 == 0 )
    log2 = log(2);  /* Only want to compute this once. */
  
  for ( i=0; i < array->size; i++ )
    {
      if ( array->data[i] >= 0 )
        sum += array->data[i];
      else
        sum -= array->data[i];
    }
  return 100.0 * log(1 + (double) sum / array->size) / log2;
}

float mean( int_array arr )
{
  int i;
  double sum = 0.0;
  for ( i=0; i < arr->size; i++ )
    {
      sum += arr->data[i];
    }
  return sum / arr->size;
}

float std( int_array arr, float mean )
{
  double sumsq = 0.0;
  int i;
  for ( i=0; i < arr->size; i++ )
    {
      sumsq += (arr->data[i] - mean) * (arr->data[i] - mean);
    }
  return sqrt( sumsq / (arr->size - 1) );
}

long sum( int_array array )
{
  int i;
  long sum = 0L;
  for ( i=0; i < array->size; i++ )
    {
      sum += array->data[i];
    }
  return sum;
}

/* weighted mean */
float wmean( int_array arr, int_array weights, long weightsum )
{
  int i;
  double sum = 0.0;
  for ( i=0; i < arr->size; i++ )
    {
      sum += ((double) arr->data[i]) * weights->data[i];
    }
  return sum / weightsum;
}

/* weighted std */
float wstd( int_array arr, float mean, int_array weights, long weightsum )
{
  double sumsq = 0.0;
  int i;
  for ( i=0; i < arr->size; i++ )
    {
      sumsq += ((double) (arr->data[i] - mean)) * (arr->data[i] - mean)
               * weights->data[i];
    }
  return sqrt( sumsq / weightsum );
}


float meanf( float_array arr )
{
  int i;
  double sum = 0.0;
  for ( i=0; i < arr->size; i++ )
    {
      sum += arr->data[i];
    }
  return (float) (sum / arr->size);
}

float stdf( float_array arr, float mean )
{
  double sumsq = 0.0;
  int i;
  for ( i=0; i < arr->size; i++ )
    {
      sumsq += (arr->data[i] - mean) * (arr->data[i] - mean);
    }
  return (float) sqrt( sumsq / (arr->size - 1) );
}

float stdr( real_array arr, fftw_real mean )
{
  double sumsq = 0.0;
  int i;
  for ( i=0; i < arr->size; i++ )
    {
      sumsq += (arr->data[i] - mean) * (arr->data[i] - mean);
    }
  return (float) sqrt( sumsq / (arr->size - 1) );
}

